﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using System.Diagnostics;
using System.Collections.Generic;
using Microsoft.Xna.Framework.Audio;

namespace GE.Audio
{
    /// <summary>
    /// Handle a playlist.
    /// </summary>
    class Playlist
    {
        /// <summary>
        /// List of sound composing the playlist
        /// </summary>
        List<SoundEffectInstance> _listSound;

        /// <summary>
        /// Id of the current sound played
        /// </summary>
        int _iCurrentSoundPlayed;

        /// <summary>
        /// Flag set when the playlist looped
        /// </summary>
        bool _bLoopPlaylist;

        /// <summary>
        /// Constructor
        /// </summary>
        public Playlist()
        {
            _listSound = new List<SoundEffectInstance>();
            _iCurrentSoundPlayed = -1;
        }

        /// <summary>
        /// Add a sound to the playlist
        /// </summary>
        /// <param name="sound">sound object</param>
        /// <returns></returns>
        public int addSound(SoundEffectInstance sound)
        {
            _listSound.Add(sound);
            return _listSound.Count - 1;
        }

        /// <summary>
        /// Stop the playlist
        /// </summary>
        public void stop()
        {
            if (_iCurrentSoundPlayed == -1) return;
            _listSound[_iCurrentSoundPlayed].Stop(true);
            _iCurrentSoundPlayed = -1;
        }

        /// <summary>
        /// Pause the playlist
        /// </summary>
        public void pause()
        {
            if (_iCurrentSoundPlayed == -1) return;
            _listSound[_iCurrentSoundPlayed].Pause();
        }

        /// <summary>
        /// Play or resume the playlist
        /// </summary>
        public void play()
        {
            if (_listSound.Count == 0) return;
            _iCurrentSoundPlayed = 0;
            _listSound[_iCurrentSoundPlayed].Play();
        }

        /// <summary>
        /// Play or resume the playlist
        /// </summary>
        /// <param name="idSound">Id of the first sound to play</param>
        public void play(int idSound)
        {
            Debug.Assert(idSound >= 0);
            Debug.Assert(idSound < _listSound.Count);

            _iCurrentSoundPlayed = idSound;
            _listSound[_iCurrentSoundPlayed].Play();
        }

        /// <summary>
        /// Play or resume a playlist
        /// </summary>
        /// <param name="idSound">Id of the first sound to play</param>
        /// <param name="loop">Flag to loop the playlist</param>
        public void play(int idSound, bool loop)
        {
            play(idSound);
            _bLoopPlaylist = loop;
        }

        /// <summary>
        /// Update the playlist
        /// </summary>
        public void update()
        {
            if (_iCurrentSoundPlayed == -1) return;
            if (_listSound[_iCurrentSoundPlayed].State == SoundState.Paused || 
                _listSound[_iCurrentSoundPlayed].State == SoundState.Playing) return;

            //_iCurrentSoundPlayed++;
            if (_bLoopPlaylist)
            {
                _iCurrentSoundPlayed = (_iCurrentSoundPlayed + 1) % _listSound.Count;
            }
            else
            {
                _iCurrentSoundPlayed++;
                if (_iCurrentSoundPlayed >= _listSound.Count)
                {
                    _iCurrentSoundPlayed = -1;
                    return;
                }
            }

            _listSound[_iCurrentSoundPlayed].Play();
        }
    }
}
